#!/bin/bash

# Git仓库操作函数
# 参数: $1 - 仓库URL, $2 - 目标目录
git_repo_sync() {
    local repo_url="$1"
    local target_dir="$2"
    local exit_code=0

    # 参数校验
    if [[ -z "$repo_url" || -z "$target_dir" ]]; then
        echo "[ERROR] 参数缺失: 必须提供仓库URL和目标目录"
        return 1
    fi

    # 创建目录（如果不存在）
    mkdir -p "$target_dir" || {
        echo "[ERROR] 无法创建目录: $target_dir"
        return 1
    }

    # 进入目录（子shell避免影响主进程）
    (
        cd "$target_dir" || {
            echo "[ERROR] 无法进入目录: $target_dir"
            exit 1
        }

        # 判断是否为Git仓库
        if git rev-parse --is-inside-work-tree >/dev/null 2>&1; then
            echo "[INFO] 更新仓库: $repo_url => $target_dir"
            if ! git pull --progress "$repo_url"; then
                echo "[ERROR] 拉取失败: $repo_url"
                exit 1
            fi
        else
            # 检查目录是否为空（允许隐藏文件）
            if [ "$(ls -A . 2>/dev/null)" ]; then
                echo "[ERROR] 目录非空: $target_dir"
                exit 1
            fi

            # 执行LFS克隆
            echo "[INFO] 开始克隆: $repo_url => $target_dir"
            if git lfs clone --progress --depth 1 --single-branch "$repo_url" .; then
                echo "[SUCCESS] 克隆完成: $repo_url"
            else
                echo "[ERROR] 克隆失败: $repo_url"
                exit 1
            fi
        fi
    )
    exit_code=$?

    return $exit_code
}

if [ -n "${git_model_repo}" ]; then
    git_repo_sync ${git_model_repo} ${git_model_dir}
fi
if [ -n "${git_dataset_repo}" ]; then
    git_repo_sync ${git_dataset_repo} ${git_dataset_dir}
fi
